<?php
session_start();
require_once '../config/db.php';
require_once '../config/config.php';

// Set error reporting
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', '../logs/mpesa_errors.log');

// Create logs directory if it doesn't exist
if (!file_exists('../logs')) {
    mkdir('../logs', 0777, true);
}

// Log the callback for debugging
$logFile = '../logs/mpesa_callback.log';
try {
    file_put_contents($logFile, date('Y-m-d H:i:s') . " - Callback received\n", FILE_APPEND);
} catch (Exception $e) {
    error_log("Failed to write to log file: " . $e->getMessage());
}

// Get the callback data
$rawData = file_get_contents('php://input');
if ($rawData === false) {
    error_log("Failed to read input data");
    exit();
}

$callbackData = json_decode($rawData, true);
if (json_last_error() !== JSON_ERROR_NONE) {
    error_log("JSON decode error: " . json_last_error_msg());
    exit();
}

try {
    file_put_contents($logFile, "Callback data: " . json_encode($callbackData) . "\n", FILE_APPEND);
} catch (Exception $e) {
    error_log("Failed to write callback data to log: " . $e->getMessage());
}

// Verify the callback
if (!isset($callbackData['Body']['stkCallback'])) {
    try {
        file_put_contents($logFile, "Invalid callback format\n", FILE_APPEND);
    } catch (Exception $e) {
        error_log("Failed to write invalid format message: " . $e->getMessage());
    }
    exit();
}

$stkCallback = $callbackData['Body']['stkCallback'];
$resultCode = $stkCallback['ResultCode'] ?? null;
$resultDesc = $stkCallback['ResultDesc'] ?? 'Unknown error';
$merchantRequestID = $stkCallback['MerchantRequestID'] ?? '';
$checkoutRequestID = $stkCallback['CheckoutRequestID'] ?? '';

// Log the result
try {
    file_put_contents($logFile, "Result Code: " . $resultCode . "\n", FILE_APPEND);
    file_put_contents($logFile, "Result Description: " . $resultDesc . "\n", FILE_APPEND);
} catch (Exception $e) {
    error_log("Failed to write result to log: " . $e->getMessage());
}

if ($resultCode == 0) {
    // Payment was successful
    $callbackMetadata = $stkCallback['CallbackMetadata']['Item'] ?? [];
    
    // Extract payment details
    $amount = 0;
    $mpesaReceiptNumber = '';
    $transactionDate = '';
    $phoneNumber = '';
    
    foreach ($callbackMetadata as $item) {
        switch ($item['Name']) {
            case 'Amount':
                $amount = $item['Value'] ?? 0;
                break;
            case 'MpesaReceiptNumber':
                $mpesaReceiptNumber = $item['Value'] ?? '';
                break;
            case 'TransactionDate':
                $transactionDate = $item['Value'] ?? '';
                break;
            case 'PhoneNumber':
                $phoneNumber = $item['Value'] ?? '';
                break;
        }
    }
    
    try {
        // Begin transaction
        $pdo->beginTransaction();
        
        // Update the payment status in your database
        $stmt = $pdo->prepare("
            UPDATE credit_payments 
            SET 
                status = 'completed',
                mpesa_receipt = ?,
                transaction_date = ?,
                updated_at = NOW()
            WHERE 
                checkout_request_id = ? 
                AND status = 'pending'
        ");
        
        $stmt->execute([
            $mpesaReceiptNumber,
            $transactionDate,
            $checkoutRequestID
        ]);
        
        $updatedRows = $stmt->rowCount();
        
        // If no rows were affected, log it
        if ($updatedRows === 0) {
            error_log("No pending payment found for checkout_request_id: " . $checkoutRequestID);
        } else {
            file_put_contents($logFile, "Updated {$updatedRows} payment record(s) for M-Pesa receipt: {$mpesaReceiptNumber}\n", FILE_APPEND);
        }
        
        $pdo->commit();
        file_put_contents($logFile, "Payment processed successfully\n", FILE_APPEND);
        
        // Send email notification after successful payment
        try {
            // Get customer details from the payment record
            $stmt = $pdo->prepare("
                SELECT DISTINCT 
                    c.id as customer_id,
                    c.name, 
                    c.email, 
                    c.phone
                FROM credit_payments cp
                JOIN customers c ON cp.customer_id = c.id
                WHERE cp.checkout_request_id = ? 
                AND cp.status = 'completed'
                LIMIT 1
            ");
            $stmt->execute([$checkoutRequestID]);
            $customer = $stmt->fetch();
            
            if ($customer && !empty($customer['email'])) {
                // Calculate updated balance for this customer
                $stmt = $pdo->prepare("
                    SELECT 
                        COALESCE(SUM(ct.amount), 0) as total_credit,
                        COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) as total_paid,
                        COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as total_waived,
                        COALESCE(SUM(ct.amount), 0) - COALESCE(SUM(CASE WHEN cp.is_waived = 0 OR cp.is_waived IS NULL THEN cp.amount ELSE 0 END), 0) - COALESCE(SUM(CASE WHEN cp.is_waived = 1 THEN cp.amount ELSE 0 END), 0) as remaining_balance
                    FROM credit_transactions ct
                    LEFT JOIN credit_payments cp ON ct.id = cp.credit_transaction_id
                    WHERE ct.customer_id = ?
                ");
                $stmt->execute([$customer['customer_id']]);
                $balance = $stmt->fetch();
                
                // Get payment details for this transaction
                $stmt = $pdo->prepare("
                    SELECT SUM(amount) as paid_amount 
                    FROM credit_payments 
                    WHERE checkout_request_id = ? AND status = 'completed'
                ");
                $stmt->execute([$checkoutRequestID]);
                $paymentInfo = $stmt->fetch();
                
                // Send payment confirmation email
                require_once '../includes/email_functions.php';
                require_once '../includes/email_templates.php';
                
                $subject = "Payment Confirmation - Receipt #" . $mpesaReceiptNumber;
                $emailBody = createPaymentConfirmationEmail(
                    $customer['name'],
                    $paymentInfo['paid_amount'] ?? $amount,
                    $mpesaReceiptNumber,
                    $balance['remaining_balance'] ?? 0,
                    $transactionDate
                );
                
                $emailSent = sendEmail($customer['email'], $subject, $emailBody);
                
                if ($emailSent) {
                    file_put_contents($logFile, "Payment confirmation email sent to: " . $customer['email'] . "\n", FILE_APPEND);
                } else {
                    file_put_contents($logFile, "Failed to send payment confirmation email to: " . $customer['email'] . "\n", FILE_APPEND);
                }
            } else {
                file_put_contents($logFile, "No customer email found for checkout_request_id: " . $checkoutRequestID . "\n", FILE_APPEND);
            }
        } catch (Exception $emailError) {
            file_put_contents($logFile, "Error sending payment confirmation email: " . $emailError->getMessage() . "\n", FILE_APPEND);
        }
        
    } catch (PDOException $e) {
        $pdo->rollBack();
        error_log("Database error: " . $e->getMessage());
        file_put_contents($logFile, "Database error: " . $e->getMessage() . "\n", FILE_APPEND);
    }
} else {
    // Payment failed
    try {
        // Begin transaction
        $pdo->beginTransaction();
        
        // Update the payment status to failed
        $stmt = $pdo->prepare("
            UPDATE credit_payments 
            SET 
                status = 'failed',
                failure_reason = ?,
                updated_at = NOW()
            WHERE 
                checkout_request_id = ? 
                AND status = 'pending'
        ");
        
        $stmt->execute([
            $resultDesc,
            $checkoutRequestID
        ]);
        
        // If no rows were affected, log it
        if ($stmt->rowCount() === 0) {
            error_log("No pending payment found for checkout_request_id: " . $checkoutRequestID);
        }
        
        $pdo->commit();
        file_put_contents($logFile, "Payment failed: " . $resultDesc . "\n", FILE_APPEND);
        
    } catch (PDOException $e) {
        $pdo->rollBack();
        error_log("Database error: " . $e->getMessage());
        file_put_contents($logFile, "Database error: " . $e->getMessage() . "\n", FILE_APPEND);
    }
}

// Send response to M-Pesa
header('Content-Type: application/json');
echo json_encode(['ResultCode' => 0, 'ResultDesc' => 'Success']); 